#include "loginscreen.h"
#include "configureservers.h"
#include "ui_loginscreen.h"

#include <QMessageBox>
#include <QFile>

LoginScreen::LoginScreen(bool noDesktop, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::LoginScreen)
{
	QIcon icon(":/librework.icns");
	setWindowIcon( icon );
	settings = new ClientSettings();
	session = new ClientSession( settings, this );
	connect( session, SIGNAL(started()), this, SLOT(sessionStarted()));
	connect( session, SIGNAL(finished()), this, SLOT(sessionFinished()));
	ui->setupUi(this);
	qApp->setStyleSheet("Widget{background-image: url(:/fondo_login.png);}");

	proxyProcess = 0;

	if ( noDesktop )
	{
		setWindowFlags( Qt::CustomizeWindowHint );
		QSize desktopSize = QGuiApplication::primaryScreen()->size();
		QSize widgetSize = size();
		move( (desktopSize.width() - widgetSize.width() ) / 2, (desktopSize.height() - widgetSize.height()) / 2 );
	} else {
		ui->powerOffButton->hide();
	}

	session->launchRemovableManagerProcess();

	QString login = settings->getLastLogin();

	if ( ! login.isEmpty() )
	{
		ui->loginLineEdit->setText(login);
		ui->passwordLineEdit->setFocus();
	}

/*
    if ( FREE_VERSION == 1 )
    {
        ui->serverLabel->setVisible(false);
        ui->serverComboBox->setItemText(ui->serverComboBox->currentIndex(), DEFAULT_SERVER);
        ui->serverComboBox->setVisible(false);
        ui->toolButton->setVisible(false);
    } else {
        serversChanged();
    }

    if ( VERSION_TYPE == 0 )
    {
        ui->editionLabel->setText(tr("Free Edition"));
    }

    if ( VERSION_TYPE == 1 )
    {
        ui->editionLabel->setText(tr("Premium Edition"));
    }

    if ( VERSION_TYPE == 2 )
    {
        ui->editionLabel->setText(tr("Business Edition"));
    }
*/
	serversChanged();
	connect(ui->serverComboBox->lineEdit(), SIGNAL(returnPressed()), this, SLOT(on_loginButton_clicked()));
}

LoginScreen::~LoginScreen()
{
	delete settings;
	delete ui;
}

void LoginScreen::forceTranslations()
{
	QString ok = tr("OK");
	Q_UNUSED(ok);
	QString cancel = tr("Cancel");
	Q_UNUSED(cancel);
}

void LoginScreen::on_toolButton_clicked()
{
    ConfigureServers * configureServers = new ConfigureServers(this, settings);
    connect(configureServers, SIGNAL(accepted()), this, SLOT(serversChanged()));
    configureServers->show();
}

void LoginScreen::serversChanged()
{
    ui->serverComboBox->clear();
    QStringList serverList = settings->defaultServersList() + settings->serverList();
    for (int i = 0; i < serverList.count(); i++)
    {
        ui->serverComboBox->addItem(settings->getIcon(serverList[i]), serverList[i]);
    }

    if (! settings->getLastServer().isNull())
    {
        if (! settings->serverList().contains(settings->getLastServer()))
        {
            ui->serverComboBox->addItem(settings->getLastServer());
        }
        ui->serverComboBox->setCurrentIndex( ui->serverComboBox->findText( settings->getLastServer(), Qt::MatchExactly ) );
    }
}

QStringList LoginScreen::readFile(QString fileName)
{
	QStringList ret;
	QFile file( fileName );
	file.open( QIODevice::ReadOnly );
	QTextStream textStream(&file);
	while (! textStream.atEnd())
	{
		QString line = textStream.readLine();
		if ( ! line.isEmpty() && ! line.startsWith('#') )
		{
			ret.append(line.simplified());
		}
	}
        file.close();
	return ret;
}

void LoginScreen::on_loginButton_clicked()
{
    QString username = ui->loginLineEdit->text();
    QString password = ui->passwordLineEdit->text();
    server = ui->serverComboBox->currentText();

    QString whiteListFileName = settings->getUserWhiteList( server );
    QStringList whiteList;
    if ( ! whiteListFileName.isEmpty() )
    {
        whiteList = readFile( whiteListFileName );
    }
    if ( whiteList.count() > 0 )
    {
        if ( ! whiteList.contains( username, Qt::CaseInsensitive ) )
        {
            QMessageBox::critical(this, tr("Bad user name"), tr("The user name must be in the white list"));
            return;
        }
    }

    QString blackListFileName = settings->getUserBlackList( server );
    QStringList blackList;
    if ( ! blackListFileName.isEmpty() )
    {
        blackList = readFile( blackListFileName );
    }
    if ( blackList.count() > 0 )
    {
        if ( blackList.contains( username, Qt::CaseInsensitive ) )
        {
            QMessageBox::critical(this, tr("Bad user name"), tr("This user name is in the black list for this server"));
            return;
        }
    }

    if (username.isEmpty())
    {
        QMessageBox::critical(this, tr("Bad user name"), tr("The login field must not be empty"));
        return;
    }

    if (password.isEmpty())
    {
        QMessageBox::critical(this, tr("Bad password"), tr("The password field must not be empty"));
        return;
    }

    if (server.isEmpty())
    {
        QMessageBox::critical(this, tr("Bad host name"), tr("The server field must not be empty"));
        return;
    }

    connPort = 0;
	bool useProxy = false; // By default, don't use  internal proxy for Librework connections
	hostName = server; // By default, use  serverComboBox text value
    QString proxyName;
    if ( settings->serverList().contains( server ) || settings->defaultServersList().contains( server ) )
    {
        if (settings->getType( server ) != "Librework") // In case of RDP connection, avoid local proxy
        {
            hostName = settings->getServer( server );
            useProxy = false;
        } else {
            proxyName = settings->getServer( server );
        }
        connPort = settings->getPort( server );
    } else {
        proxyName = server.section( ":", 0, 0 );
        connPort = server.section( ":", 1, 1 ).toInt();
        if ( connPort == 0 )
        {
			connPort = RDP_DEFAULT_PORT;
        }
    }

    if ( useProxy )
    {
        proxyProcess = new QProcess( this );
        QStringList arguments;
        arguments += proxyName;
        arguments += QString::number(connPort);
        arguments += username;
        QString command;
#ifdef Q_OS_WIN
        command = "Librework-Client-Proxy.exe";
#else
        command = "Librework-Client-Proxy";
#endif
        proxyProcess->start( command, arguments );
        proxyProcess->waitForStarted();
        proxyProcess->waitForReadyRead();
        QByteArray proxyOutput = proxyProcess->readAllStandardOutput();
        if ( proxyOutput.startsWith( "READY:" ) )
        {
            QByteArray connPortArray = proxyOutput.replace("READY:" , "");
            connPort = connPortArray.trimmed().toInt();
        }

        if ( connPort <= 0 || proxyProcess->state() == QProcess::NotRunning )
        {
			QMessageBox::critical(this, tr("Internal error"), tr("Unable to bind a port on localhost"));
			delete proxyProcess;
			return;
        }

        qDebug( QString("Connecting to proxy on localhost, port: " + QString::number(connPort)).toUtf8());
		session->launchProcess( ui->loginLineEdit->text(), ui->passwordLineEdit->text(), ui->serverComboBox->currentText(), connPort );
    } else {
        qDebug("Connecting directly to remote server");
		session->launchProcess( ui->loginLineEdit->text(), ui->passwordLineEdit->text(), ui->serverComboBox->currentText() );
    }
}

void LoginScreen::sessionStarted()
{

    ui->passwordLineEdit->setText("");
    hide();
}

void LoginScreen::sessionFinished()
{
	if ( proxyProcess != 0 )
	{
		proxyProcess->kill();
		proxyProcess->waitForFinished();
		delete proxyProcess;
		proxyProcess = 0;
	}

	show();
	activateWindow();
	raise();
}

void LoginScreen::on_passwordLineEdit_returnPressed()
{
    on_loginButton_clicked();
}

void LoginScreen::on_loginLineEdit_returnPressed()
{
    on_loginButton_clicked();
}

void LoginScreen::on_serverComboBox_currentIndexChanged(const QString &selected)
{
    if (settings->serverList().contains(selected))
    {
        QString currentLogin = settings->getLogin(selected);
        if (! selected.isEmpty() && ! currentLogin.isEmpty())
        {
            ui->loginLineEdit->setText( currentLogin );
        }
    }
}

void LoginScreen::on_powerOffButton_clicked()
{
	close();
}
