#include "command.h"

Command::Command(QString username, QString password, QString host, int port, ClientSettings * settings, QString connName, QWidget * containerWidget, bool fullWidth) : QObject()
{
	this->fullWidth = fullWidth;
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
    command = rdpCommand();

    if ( command == "rdesktop" )
    {
        if (settings->getSoundRedirect(connName) && FREE_VERSION == 0)
        {
            arguments += "-r";
            arguments += "sound:local";
        }
        arguments += oldFreerdpArguments( username, password, host, port, settings, connName, containerWidget );
    }

    if ( command == "xfreerdp3" )
    {
        if ( newFreerdpCommand() )
        {
            arguments += newFreerdpArguments( username, password, host, port, settings, connName, containerWidget );
        } else {
            arguments += "--ignore-certificate";
            arguments += "--sec";
            QString connType = settings->getType( connName );
            QString security = "rdp";
            if (connType == "RDP TLS")
                security = "tls";
            if (connType == "RDP NLA")
                security = "nla";
            arguments += security;
            if (settings->getSoundRedirect(connName) && FREE_VERSION == 0)
            {
                arguments += "--plugin";
                arguments += "rdpsnd";
            }
            if (settings->getPrintersRedirect(connName) && FREE_VERSION == 0)
            {
                arguments += "--plugin";
                arguments += "rdpdr";
                arguments += "--data";
                arguments += "printer";
            }
            arguments += oldFreerdpArguments( username, password, host, port, settings, connName, containerWidget );
        }
    }
#endif

#ifdef Q_OS_WIN
#if USE_MSTSC
	/*
    arguments.clear();
    command = "cmdkey.exe";
	arguments += "/generic:TERMSRV/" + host; // + ":" + QString::number( port );
	//arguments += "/user:" + username;
	arguments += "/user:" + settings->getDomain( connName ) + "\\" + username;
	arguments += "/pass:" + password;
	qDebug("Setting credentials: %s", QString( command + " " + arguments.join(" ")).toUtf8().data());
    int res = QProcess::execute(command, arguments);
    if ( res < 0 )
		QMessageBox::critical( 0, tr("Credentials"), tr("Unable to set credentials") );
	*/
    arguments.clear();
    command = "mstsc.exe";
    arguments = mstscArguments( username, password, host, port, settings, connName );
#else
    if (QFile::exists("wfreerdp.exe"))
    {
        command = "wfreerdp.exe";
    } else if (QFile::exists(QString(qgetenv("programfiles")) + "\\Librework Client\\wfreerdp.exe")) {
        command = QString(qgetenv("programfiles")) + "\\Librework Client\\wfreerdp.exe";
    } else if (QFile::exists(QString(qgetenv("programfiles(x86)")) + "\\Librework Client\\wfreerdp.exe")) {
        command = QString(qgetenv("programfiles(x86)")) + "\\Librework Client\\wfreerdp.exe";
    } else if (QFile::exists(QString(qgetenv("programfiles")) + "\\FreeRDP\\wfreerdp.exe")) {
        command = QString(qgetenv("programfiles")) + "\\FreeRDP\\wfreerdp.exe";
    } else if (QFile::exists(QString(qgetenv("programfiles(x86)")) + "\\FreeRDP\\wfreerdp.exe")) {
        command = QString(qgetenv("programfiles(x86)")) + "\\FreeRDP\\wfreerdp.exe";
    } else {
        command = "wfreerdp.exe";
    }
    arguments += newFreerdpArguments( username, password, host, port, settings, connName, containerWidget );
#endif
#endif

#ifdef Q_OS_MAC
    pid_t pid = (pid_t) QApplication::applicationPid();
    char pathBuf[PROC_PIDPATHINFO_MAXSIZE];

    int ret = proc_pidpath (pid, pathBuf, sizeof(pathBuf));
    if ( ret <= 0 ) // Something weird happened, try with "MacFreeRDP"
    {
        command = "MacFreeRDP";
     } else {
        QFileInfo fileInfo(pathBuf);
        command = QFileInfo(fileInfo.absolutePath() + "/../Resources/MacFreeRDP.app/Contents/MacOS/MacFreeRDP").canonicalFilePath();
        if (! QFile::exists(command))
        {
            command = "/Applications/MacFreeRDP.app/Contents/MacOS/MacFreeRDP";
            if (! QFile::exists(command))
            {
                command = "MacFreeRDP";
            }
        }
    }
    arguments += newFreerdpArguments( username, password, host, port, settings, connName, containerWidget );
#endif
}

QString Command::getCommand()
{
	return command;
}

QStringList Command::getArguments()
{
	return arguments;
}

bool Command::newFreerdpCommand()
{
    bool res = false;
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
    QProcess * process = new QProcess();
    QStringList args;
    args += "/version";
    process->start("xfreerdp3", args);
    process->waitForFinished();
    while ( process->canReadLine() && ! res )
    {
        QByteArray output = process->readLine();
        res = output.startsWith("This is FreeRDP version");
        qDebug("Version line: %s", output.data());
    }
    delete process;
#endif
    return res;
}

QString Command::cryptPassword(QString password)
{
	QString ret;
#ifdef Q_OS_WIN
	DATA_BLOB DataIn;
	DATA_BLOB DataOut;
	BYTE * pbDataInput = (BYTE*)malloc(password.length() * sizeof(WCHAR));
	password.toWCharArray((wchar_t*)pbDataInput);
	DWORD cbDataInput = password.length() * sizeof(WCHAR);

	DataIn.pbData = pbDataInput;
	DataIn.cbData = cbDataInput;

	if(CryptProtectData( &DataIn, L"psw", NULL, NULL, NULL, 0, &DataOut))
	{
		char f[2];
		for(uint i = 0 ; i < DataOut.cbData ; ++i)
		{
			sprintf(f, "%x%x", DataOut.pbData[i] >> 4, DataOut.pbData[i] & 0xf);
			ret += f;
		}
	} else {
		ret = "Error: The encryption phase didn't work.";
	}

	LocalFree( DataOut.pbData );
	free(pbDataInput);
#endif
	return ret.toUpper();
}

void Command::clearMstsc(QString hostName, QString tempFile)
{
	if ( QFile::exists(tempFile) )
	{
		QFile file(tempFile);
		file.remove();
	}
	/*
	QString command = "cmdkey.exe";
	QStringList arguments;
	arguments += "/delete:TERMSRV/" + hostName; // + ":" + QString::number( connPort );
	qDebug("Removing credentials: %s", QString( command + " " + arguments.join(" ")).toUtf8().data());
	int res = QProcess::execute(command, arguments);
	if ( res < 0 )
		QMessageBox::critical( 0, tr("Credentials"), tr("Unable to delete credentials") );
	*/
}

QStringList Command::mstscArguments( QString username, QString password, QString host, int port, ClientSettings * settings, QString connName )
{
	QStringList arguments;

	QTemporaryFile rdpTempFile;

	if ( rdpTempFile.open() )
	{
		QTextStream out( &rdpTempFile );

		// Non-configurable settings
		out << "compression:i:1\r\n";
		out << "keyboardhook:i:2\r\n";
		out << "redirectclipboard:i:1\r\n";
		out << "allow font smoothing:i:1\r\n";
		out << "authentication level:i:0\r\n";
		out << "prompt for credentials:i:0\r\n";
		out << "prompt for credentials on client:i:0\r\n";
		out << "promptcredentialonce:i:0\r\n";
		out << "use redirection server name:i:0\r\n";
		out << "displayconnectionbar:i:1\r\n";
		out << "pinconnectionbar:i:0\r\n";
		out << "autoreconnection enabled:i:1\r\n";
		out << "bitmapcachepersistenable:i:1\r\n";
		out << "bitmapcachesize:i:32000\r\n";
		out << "videoplaybackmode:i:1\r\n";
		out << "gatewayhostname:s:\r\n";
		out << "gatewayusagemethod:i:4\r\n";
		out << "gatewaycredentialssource:i:4\r\n";
		out << "gatewayprofileusagemethod:i:0\r\n";
		out << "gatewaybrokeringtype:i:0\r\n";
		out << "rdgiskdcproxy:i:0\r\n";
		out << "kdcproxyname:s:\r\n";

		// Configurable settings
		out << "full address:s:" + host + ":" + QString::number( port ) + "\r\n";
		out << "session bpp:i:" + QString::number( settings->getColorDepth( connName ) ) + "\r\n";
		QString connType = settings->getType( connName );
		if ( ! connType.isEmpty() )
		{
			if (connType == "RDP")
				out << "negotiate security layer:i:0\r\n";
			if (connType == "RDP TLS")
				out << "negotiate security layer:i:0\r\n";
			if (connType == "RDP NLA")
				out << "negotiate security layer:i:1\r\n";
		}
		if ( settings->getScreenSize( connName ) == "fullscreen" )
		{
			out << "screen mode id:i:2\r\n";
			out << "desktopwidth:i:800\r\n";
			out << "desktopheight:i:600\r\n";
		} else if ( settings->getScreenSize( connName ) == "maximized" ) {
			out << "screen mode id:i:1\r\n";
			out << "desktopwidth:i:" + QString::number(qApp->primaryScreen()->availableGeometry().size().width()) + "\r\n";
			out << "desktopheight:i:" + QString::number(qApp->primaryScreen()->availableGeometry().size().height()) + "\r\n";
		} else {
			out << "screen mode id:i:1\r\n";
			out << "desktopwidth:i:" + settings->getScreenSize(connName).section( 'x', 0, 0 ) + "\r\n";
			out << "desktopheight:i:" + settings->getScreenSize(connName).section( 'x', 1, 1 ) + "\r\n";
		}
		if ( settings->getMultimonSupport( connName ) )
		{
			out << "use multimon:i:1\r\n";
		} else {
			out << "use multimon:i:0\r\n";
		}
		out << "username:s:" + username + "\r\n";
		out << "domain:s:" + settings->getDomain( connName ) + "\r\n";
		out << "password 51:b:" + cryptPassword( password ) + "\r\n";
		// Redirect drives and dynamic devices
		out << "drivestoredirect:s:";
		if ( settings->getRemovableRedirect(connName))
		{
			out << "DynamicDevices;";
		}
		QStringList driveList = settings->getDriveRedirectFolders( connName );
		if (driveList.count() > 0)
		{

			for (int i = 0; i < driveList.count(); i++)
			{
				QString drive = driveList.at(i);
				out << drive.remove("\\") + ";";
			}
		}
		out << "\r\n";
		// End of redirect drives and dynamic devices
		if ( settings->getSoundRedirect( connName ) )
		{
			out << "audiomode:i:0\r\n";
		} else {
			out << "audiomode:i:2\r\n";
		}
		if ( settings->getMicrophoneRedirect( connName ) )
		{
			out << "audiocapturemode:i:1\r\n";
		} else {
			out << "audiocapturemode:i:0\r\n";
		}
		if ( settings->getSmartCardRedirect(connName) )
		{
			out << "redirectsmartcards:i:1\r\n";
		} else {
			out << "redirectsmartcards:i:0\r\n";
		}
		if ( settings->getPrintersRedirect(connName) )
		{
			out << "redirectprinters:i:1\r\n";
		} else {
			out << "redirectprinters:i:0\r\n";
		}
		out << "redirectcomports:i:0\r\n";
		out << "redirectposdevices:i:0\r\n";
		if (settings->getWallpaper(connName))
		{
			out << "disable wallpaper:i:0\r\n";
		} else {
			out << "disable wallpaper:i:1\r\n";
		}
		if (settings->getEffects(connName))
		{
			out << "allow desktop composition:i:1\r\n";
			out << "disable menu anims:i:0\r\n";
			out << "disable themes:i:0\r\n";
			out << "disable full window drag:i:0\r\n";
		} else {
			out << "allow desktop composition:i:0\r\n";
			out << "disable menu anims:i:1\r\n";
			out << "disable themes:i:1\r\n";
			out << "disable full window drag:i:1\r\n";
		}
		out << "disable cursor setting:i:0\r\n";
		out << "remoteapplicationmode:i:0\r\n";
		out << "alternate shell:s:\r\n";
		out << "shell working directory:s:\r\n";
		rdpTempFile.close();
	} else {
		return QStringList();
	}

	rdpTempFile.copy( rdpTempFile.fileName() + ".rdp" );

	arguments += rdpTempFile.fileName() + ".rdp";

	return arguments;
}

QStringList Command::newFreerdpArguments(QString username, QString password, QString host, int port, ClientSettings * settings, QString connName, QWidget * containerWidget)
{
	QStringList arguments;

	arguments += "/network:auto";
	arguments += "+smart-sizing";
	arguments += "+clipboard";
	arguments += "+fonts";
	arguments += "/cert:ignore";
	arguments += "/bpp:" + QString::number( settings->getColorDepth( connName ) );
	QString connType = settings->getType( connName );
	qDebug() << connType.toUtf8().data();
	if ( ! connType.isEmpty() )
	{
		QString security = "rdp";
		if (connType == "RDP TLS")
			security = "tls";
		if (connType == "RDP NLA")
			security = "nla";
		arguments += "/sec:" + security;
	}
	arguments += "/u:" + username;
	arguments += "/p:" + password;
	QString domain = settings->getDomain( connName );
	if (! domain.isEmpty())
	{
		arguments += "/d:" + domain;
	}

	if ( containerWidget != 0)
	{
		arguments += "/parent-window:" + QString::number((long)containerWidget->winId());
	}

	if (settings->getScreenSize( connName ) == "fullscreen")
	{
		arguments += "+grab-keyboard";
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
		arguments += "-decorations";
		//arguments += "-floatbar";
		arguments += "/floatbar:sticky:off,default:hidden,show:window";
		//if ( ! fullWidth )
		//{
		//	arguments += "/f";
		//}
#endif
		if ( ! fullWidth )
		{
#ifdef Q_OS_WIN
			arguments += "/f";
#endif
#ifdef Q_OS_MAC
			arguments += "/f";
#endif
			arguments += "/h:" + QString::number( QGuiApplication::primaryScreen()->size().height() );
		} else {
			arguments += "/h:" + QString::number( QGuiApplication::primaryScreen()->size().height() - BAR_HEIGHT);
		}
		arguments += "/w:" + QString::number( QGuiApplication::primaryScreen()->size().width() );
	} else {
		QString sSize = settings->getScreenSize( connName );
		int width = 0;
		int height = 0;
		if ( settings->getScreenSize( connName ) == "maximized" )
		{
			width = containerWidget->width();
			height = containerWidget->height();
		} else {
			width = sSize.section( 'x', 0, 0 ).toInt();
			height =  sSize.section( 'x', 1, 1 ).toInt();
		}
		arguments += "/w:" + QString::number(width);
		arguments += "/h:" + QString::number(height);
	}
	if (! settings->getDriveRedirectFolder(connName).isEmpty() && ! settings->getDriveRedirect(connName).isEmpty() && FREE_VERSION == 0)
	{
		arguments += "/drive:__" + settings->getDriveRedirect(connName) + "__," + settings->getDriveRedirectFolder(connName);
	}
	if (settings->getSmartCardRedirect(connName) && FREE_VERSION == 0)
	{
		arguments += "/smartcard:0";
	}
	if (settings->getHomeRedirect(connName) && FREE_VERSION == 0)
	{
#ifdef Q_OS_WIN
		QString userHome = qgetenv("homedrive") + qgetenv("homepath");
#else
		QString userHome = qgetenv("HOME");
#endif
		arguments += "/drive:home," + userHome;
	}
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
    if (settings->getRemovableRedirect(connName))
    {
        arguments += "/drive:localmedia,/media";
    }
#endif
	QStringList driveList = settings->getDriveRedirectFolders( connName );
	if (driveList.count() > 0)
	{
		QStringList nameList;
		for (int i = 0; i < driveList.count(); i++)
		{
			QString name = driveList.at(i).section(QDir::separator(), -1, -1);
			QString newName = name;
			int j = 1;
			while (nameList.contains(newName))
			{
				newName = name + "_" + QString::number(j);
				j++;
			}
			arguments +="/drive:" + newName + "," + driveList.at(i);
			nameList += newName;
		}
	}
	if (settings->getSoundRedirect(connName) && FREE_VERSION == 0)
	{
		arguments += "/sound";
		//arguments += "/multimedia";
		//arguments += "/compression";
		//arguments += "/microphone";
		//arguments += "/audio-mode";
	}
    if (settings->getMicrophoneRedirect(connName) && FREE_VERSION == 0)
    {
        arguments += "/microphone";
    }
    if (settings->getPrintersRedirect(connName) && FREE_VERSION == 0)
    {
        arguments += "/printer";
    }
    if (settings->getWallpaper(connName))
    {
        arguments += "+wallpaper";
    } else {
        arguments += "-wallpaper";
    }
    if (settings->getEffects(connName))
    {
        arguments += "+aero";
        arguments += "+menu-anims";
        arguments += "+window-drag";
        arguments += "+themes";
    } else {
        arguments += "-aero";
        arguments += "-menu-anims";
        arguments += "-window-drag";
        arguments += "-themes";
    }
    if (settings->getMultimonSupport(connName))
    {
        arguments += "/multimon";
    }

    QStringList extraArgs = settings->getExtraArgs(connName);

    arguments += extraArgs;
    arguments += "/v:" + host + ":" + QString::number( port );

    return arguments;
}

QStringList Command::oldFreerdpArguments(QString username, QString password, QString host, int port, ClientSettings * settings, QString connName, QWidget * containerWidget)
{
    QStringList arguments;

    arguments += "-u";
    arguments += username;
    arguments += "-p";
    arguments += password;
    QString domain = settings->getDomain( connName );
    if (! domain.isEmpty())
    {
        arguments += "-d";
        arguments += domain;
    }
    arguments += "-X";
    arguments += QString::number((long)containerWidget->winId());
    if (settings->getScreenSize( connName ) == "fullscreen")
    {
        arguments += "-f";
        arguments += "-g";
        arguments += QString::number( QGuiApplication::primaryScreen()->size().width() ) + "x" + QString::number( QGuiApplication::primaryScreen()->size().height() );
    } else if ( settings->getScreenSize( connName ) == "maximized") {
        arguments += "-g";
        arguments += QString::number(containerWidget->width()) + "x" + QString::number(containerWidget->height());
    } else {
        arguments += "-g";
        arguments += settings->getScreenSize( connName );
    }
    arguments += host + ":" + QString::number( port );

    return arguments;
}

QString Command::rdpCommand()
{
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
    QProcess * process = new QProcess();
    QStringList args;
    args += "xfreerdp3";
    process->start("which", args);
    process->waitForFinished();
    QByteArray output = process->readAllStandardOutput();
    qDebug(output.data());
    delete process;

    if ( output.trimmed().endsWith("xfreerdp3" ) )
        return QString( "xfreerdp3" );

    process = new QProcess();
    args.clear();
    args += "rdesktop";
    process->start("which", args);
    process->waitForFinished();
    output = process->readAllStandardOutput();
    qDebug(output.data());
    delete process;

    if ( output.trimmed().endsWith("rdesktop" ) )
        return QString( "rdesktop" );
#endif
    return QString();
}
