#include "clientsession.h"

ClientSession::ClientSession( ClientSettings * settings, QWidget * parent, bool hideContainer ) : QObject()
{
	this->parent = parent;
	this->hideContainer = hideContainer;
	process = nullptr;
	deleteSettingsOnDestroy = false;
	if ( settings != nullptr )
	{
		this->settings = settings;
	} else {
		this->settings = new ClientSettings();
		deleteSettingsOnDestroy = true;
	}

	errorShown = false;
	container = nullptr;
	containerWidget = nullptr;
	bar = nullptr;
}

ClientSession::~ClientSession()
{
	if ( ClientAppButton::isInstalled(settings, "removablemanager", server) )
	{
		removableManagerProcess->kill();
		removableManagerProcess->waitForFinished();
		delete removableManagerProcess;
	}

	if ( deleteSettingsOnDestroy )
	{
		delete settings;
	}
}

bool ClientSession::authenticateRemoteUser(QString username, QString password, QString serverConfig)
{
	ClientSettings * settings = new ClientSettings();
	QProcess * authProcess = new QProcess();
	QString command = "xfreerdp3";
	QStringList arguments;
	arguments += "/u:" + username;
	arguments += "/p:" + password;
	QString domain = settings->getDomain( serverConfig );
	if (! domain.isEmpty())
	{
		arguments += "/d:" + domain;
	}
	int port = settings->getPort( serverConfig );
	QString server = settings->getServer( serverConfig );
	arguments += "/v:" + server + ":" + QString::number( port );
	arguments += "/auth-only";
	arguments += "/cert:ignore";

	QString connType = settings->getType( serverConfig );
	if ( ! connType.isEmpty() )
	{
		QString security = "rdp";
		if (connType == "RDP TLS")
			security = "tls";
		if (connType == "RDP NLA")
			security = "nla";
		arguments += "/sec:" + security;
        }
	authProcess->start( command, arguments );
	authProcess->waitForFinished();
	bool ret = false;
	if ( authProcess->exitStatus() == QProcess::NormalExit )
	{
		ret = (authProcess->exitCode() == 0);
	}
	delete authProcess;

	return ret;
}

void ClientSession::launchRemovableManagerProcess()
{
	if ( ClientAppButton::isInstalled(settings, "removablemanager", server) )
	{
		removableManagerProcess = new QProcess();
		removableManagerProcess->start("/usr/bin/librework-removablemanager");
	}
}

void ClientSession::showContainer()
{
	mutex.lock();
	if ( container != nullptr  && containerWidget != nullptr )
	{
		container->show();
		containerWidget->show();
	}
	mutex.unlock();
}

void ClientSession::refreshConnection()
{
	if ( ! username.isEmpty() && ! password.isEmpty() && ! server.isEmpty() )
	{
		disconnect(process, SIGNAL(finished(int)), this, SLOT(finished(int)));
		disconnect( container, SIGNAL(closeRequested()), this, SLOT(finish()));
		if ( bar != nullptr )
		{
                	disconnect( bar, SIGNAL(disconnectSession()), this, SLOT(finish()));
			delete bar;
			bar = nullptr;
		}
		process->kill();
		process->waitForFinished();
		delete process;
		delete containerWidget;
		delete container;

		launchProcess( username, password, server, connPort );
	}
}

void ClientSession::launchProcess(QString username, QString password, QString server, int connPort)
{
	this->username = username;
	this->password = password;
	this->server = server;
	this->connPort = connPort;

	connect( this, SIGNAL(containerCreated()), this, SLOT( launchProcess() ) );

	mutex.lock();
	container = new Container();
	container->setWindowIcon( QIcon(":librework.png") );
	mutex.unlock();
	connect( container, SIGNAL(closeRequested()), this, SLOT(finish()));

	QString wSize = settings->getScreenSize( server );
	qDebug("Done.");
	if (wSize == "fullscreen")
	{
		container->setFixedSize( QGuiApplication::primaryScreen()->size() );
		container->setWindowState(Qt::WindowFullScreen);
		container->setWindowFlags(Qt::FramelessWindowHint);
	} else if ( wSize == "maximized" ) {
		if ( ! hideContainer )
		{
			container->showMaximized();
		}
	} else {
		int width = wSize.section( 'x', 0, 0 ).toInt();
		int height = wSize.section( 'x', 1, 1 ).toInt();
		container->setFixedSize(width, height);
	}
	if ( ! hideContainer )
	{
		container->show();
	}
	qApp->processEvents();
	QTimer::singleShot(100, this, SLOT(launchProcess()));
	qApp->processEvents();
}

void ClientSession::launchProcess()
{
	qApp->processEvents();

	bool fullWidth = true;

	if ( settings->getMultimonSupport( server ) )
	{
		fullWidth = false;
	}

	QString error;
	qDebug("Launching rdp connection");

	errorShown = false;
	long winId = 0;

	containerWidget = nullptr;

#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
	bool portKnocker = settings->getPortKnocker( server );
	if ( portKnocker )
	{
		QStringList arguments;
		arguments += "-X";
		arguments += "POST";
		arguments += "-F";
		arguments += "username=" + username;
		arguments += "https://" + settings->getServer( server ) + "/script.php";
		QProcess::execute( "curl", arguments );
	}

	QSize containerSize = container->size();
	qDebug("Container size: %dx%d", containerSize.width(), containerSize.height());
	containerWidget = new QWidget(container);
	containerWidget->resize(containerSize);
	containerWidget->move(QPoint(0,0));
	containerWidget->show();

	QString wSize = settings->getScreenSize( server );
	QString orientation = settings->getOrientation( server );
	bool vertical = ( orientation.toLower() == "vertical" );

	if (wSize == "fullscreen")
	{
		bar = new ToolBar( vertical, fullWidth, container, hideContainer);
		connect( bar, SIGNAL(refreshConnection()), this, SLOT(refreshConnection()));
		connect( bar, SIGNAL(minimizeWindow()), container, SLOT(showMinimized()));
		QRect screenrect = qApp->primaryScreen()->geometry();
		if ( fullWidth )
		{
			if ( vertical )
			{
				bar->setFixedSize( BAR_HEIGHT, screenrect.height() );
				containerWidget->move( BAR_HEIGHT, screenrect.top() );
			} else {
				bar->setFixedSize( screenrect.width(), BAR_HEIGHT );
				containerWidget->move( screenrect.left(), BAR_HEIGHT );
			}

		} else {
			if ( vertical )
			{
				bar->setFixedSize( 1, screenrect.height() );
				bar->move( QPoint( 0, ( screenrect.height() - BAR_WIDTH) / 2) );
			} else {
				bar->setFixedSize( screenrect.width(), 1 );
				bar->move( QPoint( ( screenrect.width() - BAR_WIDTH) / 2 , 0) );
			}

			if ( vertical )
			{
				bar->setFixedSize( 1, BAR_WIDTH );
			} else {
				bar->setFixedSize( BAR_WIDTH, 1 );
			}
			containerWidget->move( screenrect.left(), screenrect.top() );

		}

		QStringList apps = settings->appNames();

		for (int i = 0; i < apps.count(); i++)
		{
			if (ClientAppButton::isInstalled(settings, apps[i], server))
			{
				QProcess * rmProcess = 0;
				if ( apps[i] == "removablemanager" )
				{
					rmProcess = removableManagerProcess;
				}
				bar->addButton( new ClientAppButton(settings, apps[i], bar, rmProcess) );
			}
		}

		bar->show();
	}

	winId = (long) containerWidget->winId();
#endif

	process = new QProcess( this );

	int port = 0;
	if ( connPort != 0 )
	{
		port = connPort; // Proxy port
	} else {
		port = settings->getPort( server );
	}
	hostname = settings->getServer( server );
	Command command(username, password, hostname, port, settings, server, containerWidget, fullWidth );
#ifdef Q_OS_WIN
#if USE_MSTSC == 1
	tempFile = command.getArguments().at( command.getArguments().count() - 1 );
#endif
#endif
	qDebug( QString("Launching command: " + command.getCommand() + " " + command.getArguments().join(" ")).toUtf8());
	connect(process, SIGNAL(finished(int)), this, SLOT(finished(int)));

	process->start(command.getCommand(), command.getArguments(), QIODevice::NotOpen);
	process->waitForStarted( PROCESS_TIMEOUT );

	processStart = QDateTime::currentDateTime();
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
	if (wSize == "fullscreen")
	{
		connect( bar, SIGNAL(disconnectSession()), this, SLOT(finish()));
	}
#endif
	emit started();
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
	mountSmbShares( username, password );
#endif
}

void ClientSession::finish()
{
#if USE_MSTSC == 1
	Command::clearMstsc(hostname, tempFile);
#endif
	if ( process != nullptr )
	{
		process->kill();
		process->waitForFinished();
		delete process;
		process = nullptr;
	}

	finished(0);
}

void ClientSession::finished(int)
{
#if USE_MSTSC == 1
	Command::clearMstsc(hostname, tempFile);
#endif

	if ( parent != nullptr )
	{
		qDebug("Showing parent");
		parent->show();
	}

	mutex.lock();
	if ( container != nullptr)
	{
		container->closeContainer();
		delete container;
		container = nullptr;
	}
	mutex.unlock();

	QDateTime processEnd = QDateTime::currentDateTime();
	if (processStart.secsTo(processEnd) < RDP_TIMEOUT)
	{
#if USE_MSTSC != 1
		if ( ! errorShown )
		{
			QMessageBox::warning(nullptr, tr("Short time connection"), tr("Please, check your connection details and set a correct login name, password and server"));
			errorShown = true;
		}
#endif
	} else {
		settings->setLastLogin(username);
		settings->setLastServer(server);
	}

	emit finished();

	smbUmountAll();
}

void ClientSession::mountSmbShares(QString username, QString password)
{
    smbUmountAll();

    QString credentials = credentialsFile();
	QSettings smbSettings(CLIENT_CONF_DIR "/smbshares.ini", QSettings::IniFormat);

    // Mount home dir at .homepath
    QString smbHomePath = smbSettings.value("global/HomePath", QVariant()).toString();
    smbHomePath = smbHomePath.replace("%USER%", username);
    QString targetHomePath = smbSettings.value("global/SmbMountPath", QVariant()).toString();
    targetHomePath = targetHomePath.replace("%USER%", username);
    if (! smbHomePath.isEmpty() && ! targetHomePath.isEmpty() )
    {
        smbMount( credentials, smbHomePath, targetHomePath );
    }

    // Mount shares
    QStringList groups = smbSettings.childGroups();
    for ( int i = 0; i < groups.count(); i++ )
    {
        if ( groups[i].compare("global", Qt::CaseInsensitive) != 0 )
        {
            QString smbPath = smbSettings.value( groups[i] +"/Path" ).toString();
            QString targetPath = targetHomePath + "/" + groups[i];
            if ( ! smbPath.isEmpty() && ! targetPath.isEmpty() )
            {
                smbMount( credentials, smbPath, targetPath );
            }
        }
    }

    // Remove credentials file
	if ( ! credentials.isEmpty() )
	{
		QFile credsFile(credentials);
		credsFile.remove();
		credsFile.close();
	}
}

void ClientSession::smbMount(QString credentialsFile, QString smbPath, QString targetPath)
{

}

void ClientSession::smbUmountAll()
{

}

QString ClientSession::credentialsFile()
{
	return QString();
}
